const multer = require("multer");
const path = require("path");
const fs = require("fs"); // Use standard fs for mkdirSync
const fsPromises = require("fs").promises; // Async fs for deletion

// --- 1. Define the directory path ---
// From: backend/storage/middlewareStorage/uploadBlogMedia.js
// To:   backend/storage/storageMedia/blogs
const uploadDir = path.join(__dirname, "../storageMedia/blogs");

// --- 2. Ensure directory exists ---
if (!fs.existsSync(uploadDir)) {
  fs.mkdirSync(uploadDir, { recursive: true });
}

// --- 3. Multer Configuration ---
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const ext = path.extname(file.originalname).toLowerCase();
    const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
    cb(null, `blog-${uniqueSuffix}${ext}`);
  },
});

const fileFilter = (req, file, cb) => {
  if (file.mimetype.startsWith("image")) {
    cb(null, true);
  } else {
    cb(new Error("Not an image! Please upload only images."), false);
  }
};

const upload = multer({
  storage,
  fileFilter,
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB limit (higher for blog headers)
});

exports.uploadBlogImage = upload.single("coverImage"); // Field name expected from frontend

// --- 4. Helper: Delete Old Image ---
exports.deleteOldBlogImage = async (imageName) => {
  if (!imageName) return;

  const filePath = path.join(uploadDir, imageName);

  try {
    await fsPromises.unlink(filePath);
  } catch (err) {
    // Ignore error if file doesn't exist
  }
};
